/* 
 * Copyright 2012 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.sax;

import java.util.Locale;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import de.avm.android.tr064.model.Contact;
import de.avm.android.tr064.model.ContactEmail;
import de.avm.android.tr064.model.ContactNumber;
import de.avm.android.tr064.model.PhoneBook;

/*
 * SAX-Handler for AVM Phonebook-Format
 */
public class SAXPhoneBookHandler extends DefaultHandler
{
	private boolean mInCategory = false;
	private boolean mInRealName = false;

	private PhoneBook mPhoneBook;
	private Contact mCurrentContact = null;
	private ContactNumber mCurrentContactNumber = null;
	private ContactEmail mCurrentContactEmail = null;
	private String mCurrentCategory = "";

	/**
	 * Instantiates a new sAX phone book handler.
	 * 
	 * @param phoneBook
	 *            the phone book
	 */
	public SAXPhoneBookHandler(PhoneBook phoneBook)
	{
		mPhoneBook = phoneBook;
	}

	@Override
	public void startElement(String namespaceURI, String localName,
			String qName, Attributes atts) throws SAXException
	{
		String name = localName.toLowerCase(Locale.US);
		
		if (name.equals("phonebook"))
		{
			mPhoneBook.setName(atts.getValue("name"));
		}
		else if (name.equals("contact"))
		{
			mCurrentContact = new Contact();
		}
		else if (name.equals("category"))
		{
			mCurrentCategory = "";
			mInCategory = true;
		}
		else if (name.equals("realname"))
		{
			mInRealName = true;
		}
		else if (name.equals("number"))
		{
			mCurrentContactNumber = new ContactNumber();
			
			mCurrentContactNumber.setTypeFromKey(atts.getValue("type"));
			mCurrentContactNumber.setQuickdial(atts.getValue("quickdial"));
		}
		else if (name.equals("email"))
		{
			mCurrentContactEmail = new ContactEmail();
			mCurrentContactEmail.setTypeFromKey(atts.getValue("classifier"));
		}
	}

	@Override
	public void endElement(String namespaceURI, String localName, String qName)
			throws SAXException
	{
		String name = localName.toLowerCase(Locale.US);
		
		if (name.equals("contact"))
		{
			if (mCurrentContact != null)
			{
				mPhoneBook.addContact(mCurrentContact);
				mCurrentContact = null;
			}
		}
		else if (name.equals("category"))
		{
			if (mCurrentCategory.length() > 0)
				mCurrentContact.setCategory(Integer.parseInt(mCurrentCategory));
			mInCategory = false;
		}
		else if (name.equals("realname"))
		{
			mInRealName = false;
		}
		else if (name.equals("number"))
		{
			if ((mCurrentContactNumber != null) &&
					(mCurrentContactNumber.getNumber().trim().length() > 0))
				mCurrentContact.addNumber(mCurrentContactNumber);
			mCurrentContactNumber = null;
		}
		else if (name.equals("email"))
		{
			if ((mCurrentContactEmail != null) &&
					(mCurrentContactEmail.getEmail().trim().length() > 0))
				mCurrentContact.addEmail(mCurrentContactEmail);
			mCurrentContactEmail = null;
		}
	}

	@Override
	public void characters(char ch[], int start, int length)
	{
		String str = new String(ch, start,length);
		
		if (mInCategory)
		{
			mCurrentCategory += str;
		}
		else if (mInRealName)
		{
			mCurrentContact.setRealName(mCurrentContact.getRealName() + str);			
		}
		else if (mCurrentContactNumber != null)
		{
			mCurrentContactNumber.setNumber(mCurrentContactNumber
					.getNumberRaw() + str);
		}
		else if (mCurrentContactEmail != null)
		{
			mCurrentContactEmail.setEmail(mCurrentContactEmail.getEmail() + str);
		}
	}
}
