/* 
 * Copyright 2012 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.model;

import java.util.Locale;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

/* Represents a email address in a contact */
public class ContactEmail implements Parcelable, IContactable
{
	public enum Type
	{
		PRIVATE, BUSINESS, OTHER;

		/**
		 * Gets the text resource.
		 * 
		 * @param context
		 *            the context
		 * 
		 * @return the text resource
		 */
		public String getTextResource(Context context)
		{
			int id = context.getResources()
					.getIdentifier("contact_email_" + toString()
							.toLowerCase(Locale.US), "string",
							context.getApplicationContext().getPackageName());
			
			if (id == 0) return "";
			return context.getString(id);
		}
		
		/**
		 * Gets the email type for key.
		 * 
		 * @param key
		 *            the key
		 * 
		 * @return the email type for key
		 */
		public static Type getFromKey(String key)
		{
			for (Type type : Type.values())
			{
				if (key.toLowerCase(Locale.US).equals(type.toString()
						.toLowerCase(Locale.US)))
					return type;
			}
			// default
			return OTHER;
		}
	};

	private static final String CUSTOM_LABEL = "label:";
	
	protected Type mType = Type.OTHER;
	protected String mCustomType = null;
	protected String mEmail = "";

	/**
	 * Instantiates a new contact email.
	 */
	public ContactEmail()
	{
		super();
	}
	
	public Type getType()
	{
		return mType;
	}

	/**
	 * Instantiates a new contact email.
	 * 
	 * @param in
	 *            the in
	 */
	public ContactEmail(Parcel in)
	{
		mType = Type.values()[in.readInt()];
		mCustomType = in.readString();
		mEmail = in.readString();
	}

	/**
	 * Sets the type.
	 * 
	 * @param type
	 *            the new type
	 */
	public void setType(Type type)
	{
		mType = type;
		mCustomType = null;
	}

	/**
	 * Sets the type to custom type.
	 * 
	 * @param type
	 *            the custom type
	 */
	public void setCustomType(String type)
	{
		mType = Type.OTHER;
		mCustomType = (TextUtils.isEmpty(type)) ? null : type;
	}

	/**
	 * Sets the type from phonebook's type attribute
	 * 
	 * @param key
	 *			the attribute value
	 */
	public void setTypeFromKey(String key)
	{
		mType = Type.getFromKey(key);
		if ((mType == Type.OTHER) && key.toLowerCase(Locale.US)
				.startsWith(CUSTOM_LABEL))
			mCustomType = key.substring(CUSTOM_LABEL.length());
	}
	
	/**
	 * Gets phonebook's type attribute
	 * 
	 * @param customTypeSupport
	 * 			true to generate custom types too
	 * @return
	 * 			the phonebook's type attribute value
	 */
	public String getTypeAsKey(boolean customTypeSupport)
	{
		if (customTypeSupport && (mType == Type.OTHER) &&
				!TextUtils.isEmpty(mCustomType))
			return CUSTOM_LABEL + mCustomType;
		return mType.toString().toLowerCase(Locale.US);
	}

	/**
	 * Gets the email address.
	 * 
	 * @return the email address
	 */
	public String getEmail()
	{
		return mEmail;
	}

	/**
	 * Sets the email address.
	 * 
	 * @param email
	 *            the new email address
	 */
	public void setEmail(String email)
	{
		mEmail = (email == null) ? "" : email;
	}

	public int describeContents()
	{
		return 0;
	}

	public void writeToParcel(Parcel out, int flags)
	{
		out.writeInt(mType.ordinal());
		out.writeString(mCustomType);
		out.writeString(mEmail);
	}

	public static final Parcelable.Creator<ContactEmail> CREATOR =
			new Parcelable.Creator<ContactEmail>()
	{
		public ContactEmail createFromParcel(Parcel in)
		{
			return new ContactEmail(in);
		}

		public ContactEmail[] newArray(int size)
		{
			return new ContactEmail[size];
		}
	};

	public String getAddressDisplay()
	{
		return getEmail();
	}

	public String getTypeDisplay(Context context)
	{
		if ((mType == Type.OTHER) && !TextUtils.isEmpty(mCustomType))
			return mCustomType;
		return mType.getTextResource(context);
	}

	public String getAddress()
	{
		return getEmail();
	}
}
