/* 
 * Copyright 2015 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.fritzapp.util;

import android.os.Build;

import de.avm.fundamentals.logger.FileLog;

/**
 * Helper for OS build related stuff
 * (compatible to SDK version 3/Android 1.5)
 */
public class BuildHelper
{
	private static final String TAG = "BuildHelper"; 
	private static final int DEFAULT_RESAMPLER_VOLUME = 100; // default for all devices (0...100%)

	private static final int CALL_INTENT_WORAROUND_MINVERSION = Build.VERSION_CODES.JELLY_BEAN_MR2;

	static
	{
		dump();
	}

	public static void dump()
	{
		FileLog.d(TAG, String.format("Manufacturer: \"%s\", Model: \"%s\", SDK: %d",
                Build.MANUFACTURER, Build.MODEL, Build.VERSION.SDK_INT));
	}
	
	/**
	 * Checks if this device could be affected by SCREEN_OFF wifi issue
	 * (see isDimOnlyDevice())
	 * 
	 * @return true if workaround could be applied
	 */
	public static boolean hasDimOnlyOption()
	{
		// not needed any more?
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB)
			return false;
		if ((Build.VERSION.SDK_INT >= Build.VERSION_CODES.GINGERBREAD) &&
				(Build.MODEL.equals("Nexus One") || Build.MODEL.equals("Nexus S") ||
						Build.MODEL.equals("MB860")))
			return false;
		return true;
	}
	
	/**
	 * Checks if display should be dimmed only in calls as default
	 * 
	 * used to workaround broken audio output while SCREEN_OFF
	 * possibly a bug on some devices
	 * prevent SCREEN_OFF, dim instead
	 */
	public static boolean isDimOnlyDevice()
	{
		if (hasDimOnlyOption())
		{
			if ((Build.MODEL.startsWith("HTC ") ||
					Build.MANUFACTURER.equalsIgnoreCase("HTC")))
				return true;
	
			if (Build.MODEL.equals("Nexus One") ||
				Build.MODEL.equals("Archos5") ||
				Build.MODEL.equals("Dell Streak") ||
				Build.MODEL.equals("Nexus S") ||
				Build.MODEL.equals("MB860") ||		// Motorola Atrix
				Build.MODEL.equals("U8510"))		// Huawei Ideos X3
					return true;
		}
		return false;
	}
	
	public static boolean withRtpStreamSender() // for debugging purposes
	{
		return true;
	}

	public static boolean isSamsungGalaxyS()
	{
		return Build.MODEL.equals("GT-I9000");
	}
	
	public static boolean isSamsungGalaxy3()
	{
		return Build.MODEL.equals("GT-I5800");
	}

	public static boolean needsAudioTrackPreparation() {
		return isSamsungGalaxy3() || isSamsungGalaxyS() ||
				Build.MODEL.equals("GT-I5700") ||
				Build.MODEL.equals("GT-I5500") ||
				Build.MODEL.equals("Galaxy") ||			// i7500
				Build.MODEL.equals("GT-P1000") ||		// Galaxy Tab
				Build.MODEL.equals("Desire HD") ||
				Build.MODEL.equals("Desire Z") ||
				Build.MODEL.equals("HTC Vision");
	}
	
	private static boolean isSamsungGalaxySWithOsOlderThanFroyo()
	{
		return isSamsungGalaxyS() &&
				(Build.VERSION.SDK_INT < Build.VERSION_CODES.FROYO);
	}

	private static boolean isSamsungGalaxySWithOsNewerThanEclair()
	{
		return isSamsungGalaxyS() &&
				(Build.VERSION.SDK_INT >= Build.VERSION_CODES.FROYO);
	}
	
	public static boolean useRecordingDevice()
	{
		return !Build.MODEL.equals("smartpad");
	}
	public static boolean changeRecordStateWhenSendingDtmf()
	{
		if(!useRecordingDevice())
			return false;

		if (Build.MODEL.equals("LT30p") ||			// Sony XPERIA T
				Build.MODEL.equals("HTC One X"))	// HTC One X
			return false;
		
		return !isSamsungGalaxySWithOsNewerThanEclair();
	}
	
	public static boolean avoidSilenceAfterSpeakerModeChange()
	{
		return isSamsungGalaxySWithOsOlderThanFroyo();
	}
	
	public static boolean muteSupported()
	{
		return true;
	}
	
	public static boolean handsFreeCallingModeSupported()
	{
		if (isSamsungGalaxy3()) return false;
		
		if (isSamsungGalaxyS() &&
				((Build.VERSION.SDK_INT == Build.VERSION_CODES.FROYO) ||
				(Build.VERSION.SDK_INT == Build.VERSION_CODES.GINGERBREAD)))
			return false;

		return true;
	}

	public static boolean needsMuteHackForSpeakermodeChange()
	{
		return (handsFreeCallingModeSupported() && isSamsungGalaxyS() && 
				(Build.VERSION.SDK_INT <= Build.VERSION_CODES.GINGERBREAD));
	}
	
	public static boolean needsManualVolume()
	{
		return isSamsungGalaxyS() ||
				Build.MODEL.equals("GT-I9505") ||	// Samsung Galaxy S4
				Build.MODEL.equals("GT-I9295");		// Samsung Galaxy S4 active
	}
	
	public static boolean needsSetMode()
	{
		return isSamsungGalaxySWithOsOlderThanFroyo() ||
				Build.MODEL.equals("LG-P500") || Build.MODEL.equals("ZTE-BLADE");
	}
	
	/**
	 * Gets device specific resampler volume or returns default
	 * 
	 * @return value to be used in %
	 */
	public static int getResamplerVolume()
	{
		// return device specific value if available
		if (isSamsungGalaxyS())
			return 45;
		
		if (Build.MODEL.equals("GT-I5500") ||		// Samsung Galaxy i5500
				Build.MODEL.equals("GT-I9100") ||	// Samsung Galaxy S 2
				Build.MODEL.equals("GT-I9300") ||	// Samsung Galaxy S 3
				Build.MODEL.equals("X10i") ||		// Sony Ericsson XPERIA X10
				Build.MODEL.equals("E10i") ||		// Sony Ericsson XPERIA X10 mini
				Build.MODEL.equals("MotoMB511"))	// Motorola Flipout
			return 200;
		
		if (Build.MODEL.equals("E15i"))				// Sony Ericsson XPERIA X8
			return 180;
		
		if (Build.MODEL.equals("HTC Magic"))		// HTC Magic
			return 170;
		
		if (Build.MODEL.equals("Desire Z") ||		// HTC Desire Z (Europe)
				Build.MODEL.equals("HTC Vision") || // HTC Desire Z (non Europe)
				Build.MODEL.equals("HTC Desire"))	// HTC Desire
			return 165;
		
		if (Build.MODEL.equals("Desire HD") ||		// HTC Desire HD
				Build.MODEL.equals("HTC Hero") ||	// HTC Hero
				Build.MODEL.equals("LG-P880"))		// LG Optimus 4xHD
			return 50;
		
		if (Build.MODEL.equals("LG-P970") ||		// LG Optimus Black
				Build.MODEL.equals("LG-P990"))		// LG Optimus Speed
			return 225;
		
		if (Build.MODEL.equals("MT15i"))			// Sony Ericsson XPERIA Neo
			return 95;
		
		if (Build.MODEL.equals("XT910"))			// Motorola Razr
			return 65;

		if (Build.MODEL.equals("LT18i"))			// Sony Ericsson XPERIA Arc
			return 85;

		if (Build.MODEL.equals("LT26i"))			// Sony XPERIA S
			return 150;

		if (Build.MODEL.equals("GT-S5360") ||		// Samsung Galaxy Y
				Build.MODEL.equals("GT-S5363") ||	// Samsung Galaxy Y
				Build.MODEL.equals("GT-I8160P"))	// Samsung Galaxy Ace2
			return 250;

		if (Build.MANUFACTURER.equals("Sony") &&
				Build.MODEL.equals("C6603"))		// Sony XPERIA Z
			return 165;

		return DEFAULT_RESAMPLER_VOLUME;
	}
	
	public static boolean isCallIntentHtcDefault()
	{
		// due to a bug in Android's telephone and contacts apps since
		// 4.3 use alternative integration on all devices
		// (see https://code.google.com/p/android/issues/detail?id=58097)
		// TODO define MAXVERSION, when fixed Android version has been released
        if (Build.VERSION.SDK_INT >= CALL_INTENT_WORAROUND_MINVERSION)
			return true;
		
		return ((Build.MODEL.startsWith("HTC ") ||
				Build.MANUFACTURER.equalsIgnoreCase("HTC")));
	}
	
	/**
	 * @return true if 44.1 kHz recording has to be used
	 */
	public static boolean isRecordingWith44kHzDefault()
	{
		if (Build.MODEL.equals("Archos5") ||		// Archos 5
				Build.MODEL.equals("GT-S5360") ||	// Samsung Galaxy Y
				Build.MODEL.equals("GT-S5363") ||	// Samsung Galaxy Y
				Build.MODEL.equals("GT-S6102") ||	// Samsung Galaxy Y Duos
				Build.MODEL.equals("GT-S6802") ||	// Samsung Galaxy Ace Duos
				Build.MODEL.equals("GT-S5300"))		// Samsung Galaxy Pocket
			return true;
		
		return false;
	}
}
