/* 
 * Copyright 2012 by AVM GmbH <info@avm.de>
 *
 * This software contains free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License ("License") as 
 * published by the Free Software Foundation  (version 3 of the License). 
 * This software is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the copy of the 
 * License you received along with this software for more details.
 */

package de.avm.android.tr064.soap;

import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import android.text.TextUtils;
import de.avm.android.tr064.exceptions.BaseException;
import de.avm.android.tr064.exceptions.DataMisformatException;
import de.avm.android.tr064.model.WLANInfo;

/* Soap implementation for the message-status-Interface */
public class GetWLANStatus extends WLANSoapHelper<WLANInfo>
{
	private int mIndex;
	private String mMac = null;
	
	/**
	 * Gets WLAN info by index
	 * 
	 * @param c
	 * @param index
	 */
	public GetWLANStatus(ISoapCredentials soapCredentials, int interfaceIndex, int deviceIndex)
	{
		super(soapCredentials, interfaceIndex);
		mIndex = deviceIndex;
	}
	
	/**
	 * Gets WLAN info by MAC address
	 * 
	 * @param c
	 * @param index
	 */
	public GetWLANStatus(ISoapCredentials soapCredentials, int interfaceIndex, String mac)
	{
		super(soapCredentials, interfaceIndex);
		mMac = mac;
	}
	
	@Override
	public String getSoapMethodParameter()
	{
		return (TextUtils.isEmpty(mMac)) ?
				"<NewAssociatedDeviceIndex>" + mIndex + "</NewAssociatedDeviceIndex>" :
					"<NewAssociatedDeviceMACAddress>" + mMac + "</NewAssociatedDeviceMACAddress>";
	}

	@Override
	public WLANInfo getQualifiedResult()
			throws DataMisformatException, BaseException, IOException
	{
		String input = getSoapBody();
		WLANInfo result = new WLANInfo();
		if (TextUtils.isEmpty(mMac))
			result.setMacAdress(getValueByName("NewAssociatedDeviceMACAddress", input));
		else
			result.setMacAdress(mMac);
		result.setIpAdress(getValueByName("NewAssociatedDeviceIPAddress", input));
		String value = getValueByName("NewAssociatedDeviceAuthState", input);
		if ((value.length() > 0) && !value.equals("0")) result.setAuthState(true);
		result.setBandwidth(Integer.parseInt(getValueByName("NewX_AVM-DE_Speed", input)));
		result.setSignalStrength(Integer.parseInt(getValueByName("NewX_AVM-DE_SignalStrength", input)));
		return result;
	}
	
	/**
	 * Helper method. Get a value for a given xml-tagname
	 * 
	 * @param name
	 *            the name of the tag
	 * @param input
	 *            the input-xml-string
	 * 
	 * @return the value by name
	 */
	private String getValueByName(String name, String input)
			throws DataMisformatException
	{
		if (input.length() == 0) {
			return "";
		}
		Matcher m = Pattern.compile(name + ">(.*?)<").matcher(input);
		if (m.find()) {
			return decodeEntities(m.group(1));
		} else {
			throw new DataMisformatException(
					"Invalid Response from WLAN-Service");
		}
	}

	@Override
	public String getSoapMethod()
	{
		return (TextUtils.isEmpty(mMac)) ?
				"GetGenericAssociatedDeviceInfo" : "GetSpecificAssociatedDeviceInfo";
	}
}
