/*--------------------------------------------------------------------------*\

    CKMDEV.C      Version 1.0                                     2000 AVM

    This file contains the source for handling IoDeviceObjects for CAPI,
    here for Windows - kernel mode driver level.

\*--------------------------------------------------------------------------*/
#include "ckmdldef.h"
#include "ckmdev.h"
#include "kassert.h"


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
typedef struct _DEVICE_INFO {
    PFILE_OBJECT   pFileObj;           /* != 0 if open */
    PDEVICE_OBJECT pDevObj;
    KEVENT         hEvent;             /* for synchronous call */
} DEVICE_INFO, *PDEVICE_INFO;


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static WCHAR CapiDeviceName[] = CAPI_DEVICE_NAME;

static UNICODE_STRING UniCapiDeviceName = {
    sizeof (CapiDeviceName)-2,      /* length without terminating L'\0' */
    sizeof (CapiDeviceName),
    CapiDeviceName
};


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
NTSTATUS DeviceObjectOpen (void **hDevice) {

    NTSTATUS     status;
    PDEVICE_INFO pDev;

    assert (hDevice != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    pDev = ExAllocatePool(NonPagedPool, sizeof(DEVICE_INFO));
    assert (pDev != 0);

    status = IoGetDeviceObjectPointer (&UniCapiDeviceName,
                                       FILE_READ_DATA | FILE_WRITE_DATA,
                                       &pDev->pFileObj,
                                       &pDev->pDevObj);

    if (status == STATUS_SUCCESS) {
        assert (pDev->pFileObj != 0);
        assert (pDev->pDevObj != 0);
        assert (KeGetCurrentIrql () == PASSIVE_LEVEL);
        KeInitializeEvent (&pDev->hEvent, NotificationEvent, FALSE);
    } else {
        ExFreePool (pDev);
        pDev = 0;
    }

    *hDevice = pDev;

    return status;
}


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void DeviceObjectClose (void *hDevice) {

    PDEVICE_INFO pDev;

    assert (hDevice != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    pDev = (PDEVICE_INFO) hDevice;

    if (pDev->pFileObj != 0) {
        ObDereferenceObject (pDev->pFileObj);
    }

    ExFreePool (hDevice);
}


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static NTSTATUS SyncRequestComplete (IN PDEVICE_OBJECT pDO,
                                     IN PIRP           pIRP,
                                     IN void *         pContext) {

    PDEVICE_INFO pDev;

    #ifndef NDEBUG
        LONG OldState;
    #endif

    assert (pContext != 0);
    assert (KeGetCurrentIrql () <= DISPATCH_LEVEL);

    pDev = (PDEVICE_INFO) pContext;

    #ifndef NDEBUG
        OldState = KeSetEvent (&pDev->hEvent, 0, FALSE);
        assert (OldState == 0);
    #else
        KeSetEvent (&pDev->hEvent, 0, FALSE);
    #endif

    /*----- prevent IO manager from further IRP processing -----*/
    return STATUS_MORE_PROCESSING_REQUIRED;
}


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
NTSTATUS DeviceObjectRequest (void *hDevice,
                              unsigned long IoControlCode,
                              unsigned long InputBufferLength,
                              unsigned long OutputBufferLength,
                              void *Buffer) {

    PDEVICE_INFO       pDev;
    PIO_STACK_LOCATION pIrpLoc;
    NTSTATUS           status;
    PIRP               pIrp;

    #ifndef NDEBUG
        NTSTATUS       result;
    #endif

    assert (hDevice != 0);
    assert (KeGetCurrentIrql () == PASSIVE_LEVEL);

    pDev = (PDEVICE_INFO) hDevice;

    /*----- prepare IRP -----*/
    assert (pDev->pDevObj != 0);
    assert (pDev->pFileObj != 0);

    pIrp = IoAllocateIrp (pDev->pDevObj->StackSize, FALSE);

    if (pIrp == 0) {
        /*----- mapped onto Capi*OSResourceErr -----*/
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    pIrpLoc = IoGetNextIrpStackLocation (pIrp);
    assert (pIrpLoc != 0);

    /*----- fill the IRP -----*/
    if (   IoControlCode == IOCTL_CAPI_PUT_MESSAGE
        || IoControlCode == IOCTL_CAPI_GET_MESSAGE) {
        pIrpLoc->MajorFunction                               = IRP_MJ_INTERNAL_DEVICE_CONTROL;
        pIrpLoc->Parameters.DeviceIoControl.Type3InputBuffer = Buffer;
    } else {
        pIrpLoc->MajorFunction           = IRP_MJ_DEVICE_CONTROL;
        pIrp->AssociatedIrp.SystemBuffer = Buffer;
    }

    assert (pDev->pFileObj != 0);
    assert (pDev->pDevObj  != 0);

    pIrpLoc->Parameters.DeviceIoControl.IoControlCode      = IoControlCode;
    pIrpLoc->Parameters.DeviceIoControl.InputBufferLength  = InputBufferLength;
    pIrpLoc->Parameters.DeviceIoControl.OutputBufferLength = OutputBufferLength;
    pIrpLoc->FileObject                                    = pDev->pFileObj;
    pIrpLoc->DeviceObject                                  = pDev->pDevObj;

    /*----- synchronous call -----*/
    KeClearEvent (&pDev->hEvent);
    IoSetCompletionRoutine (pIrp, SyncRequestComplete, pDev, TRUE, TRUE, TRUE);

    status = IoCallDriver (pDev->pDevObj, pIrp);

    if (status == STATUS_PENDING) {
        #ifndef NDEBUG
            result = KeWaitForSingleObject (&pDev->hEvent, Executive, KernelMode, FALSE, 0);
            assert (result == STATUS_SUCCESS);
        #else
            KeWaitForSingleObject (&pDev->hEvent, Executive, KernelMode, FALSE, 0);
        #endif

        status = pIrp->IoStatus.Status;
    }

    assert (KeReadStateEvent (&pDev->hEvent) != 0);
    assert (status != STATUS_PENDING);
    assert (status == pIrp->IoStatus.Status);
    IoFreeIrp (pIrp);

    return status;
}


/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
